const fetch = require('node-fetch');
const https = require('https');

const PAGE_SIZE = 50; //number of items per page DONT CHANGE

//is needed for self-signed certificates
//if you use a self-signed certificate, you need to set rejectUnauthorized to false
//if you use a valid certificate, you can change it to true
const httpsAgent = new https.Agent({
  rejectUnauthorized: false,
});

//change address to your server
const server = 'https://localhost:8443';

//subsystem to query specific items can be 'pool', 'device', 'volume', etc.
const subsystem = 'device';

//get first four paramaeter from cmd username, password, and user and password for cfg change
const args = process.argv.slice(2);
if (args.length < 3) {
  console.error('Usage: node storage_capacity_api.js <username> <password> <filename.csv>');
  process.exit(1);
}
const [username, password, filename] = args;

/**
 * Fetch authentication token
 * @param {string} username
 * @param {string} password
 * @returns {Promise<string>} apiKey
*/
const fetchAuth = async (username, password) => {
  const response = await fetch(server + "/api/public/v1/auth", {
    agent: httpsAgent,
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify({
      username: username,
      password: password,
    }),
  });

  if (!response.ok) {
    throw new Error(`Failed to authenticate: ${response.statusText}`);
  }

  const apikeyData = await response.json();
  return apikeyData.data.apiKey;
}

/**
 * Fetch items with pagination
 * @param {string} apiKey
 * @returns {Promise<Array<ObjectItems>>}
*/
const fetchItems = async (apiKey) => {
  let page = 0;
  let allItems = [];
  let hasMore = true;

  while (hasMore) {
    const response = await fetch(server + `/api/public/v1/architecture/search?subsystem=${subsystem}&class=storage&type=csv&page=${page}`, {
      agent: httpsAgent,
      headers: {
        'Content-Type': 'application/json',
        'apiKey': apiKey,
      },
      method: 'GET',
    });


    if (!response.ok) {
      throw new Error(`Failed to fetch items: ${response.statusText}`);
    }

    const { data: { metadata, items } } = await response.json();
    allItems = allItems.concat(items);
    hasMore = (page + 1) * PAGE_SIZE < metadata.max && items.length > 0;
    page += 1;

  }

  return allItems;
}

/**
 * Fetch capacity data for given item IDs
 * @param {string} apiKey
 * @param {Array<string>} itemIds
 * @returns {Promise<string>} capacity data in CSV format
*/
const fetchCapacityData = async (apiKey, itemIds) => {
  const response = await fetch(server + `/api/public/v1/exporter/capacity`, {
    agent: httpsAgent,
    headers: {
      'Content-Type': 'application/json',
      'apiKey': apiKey,
    },
    method: 'POST',
    body: JSON.stringify({
      uuids: itemIds,
      format: "csv",
      subsystem: subsystem,
    }),
  });

  if (!response.ok) {
    throw new Error(`Failed to fetch capacity data: ${response.statusText}`);
  }

  const capacityData = await response.text()
  console.log(capacityData);
  return capacityData;
}

/**
 * Write capacity data to a file
 * @param {string} data
 * @param {string} filename
*/
const writeCapacityDataToFile = (data, filename) => {
  const fs = require('fs');
  fs.writeFileSync(filename, data);
  console.log(`Capacity data written to ${filename}`);
}

const main = async () => {
  try {
    const apiKey = await fetchAuth(username, password);
    const items = await fetchItems(apiKey);
    const itemIds = items.map(item => item.item_id);
    const capacityData = await fetchCapacityData(apiKey, itemIds);
    writeCapacityDataToFile(capacityData, filename);
  } catch (error) {
    console.error('Error:', error);
  }
}

main().catch(error => {
  console.error('Error:', error);
}).finally(() => {
  process.exit(0);
});

